//
// Copyright 2015-2021 by Garmin Ltd. or its subsidiaries.
// Subject to Garmin SDK License Agreement and Wearables
// Application Developer Agreement.
//

import Toybox.Communications;
import Toybox.Cryptography;
import Toybox.Lang;
import Toybox.Time.Gregorian;

using Toybox.PersistedContent;

class TpApi {

var host;
var uri;
var cb;

var body;
var usr;
var pwd;

var rlm;
var jwt;

var html_opts as {
            :method as HttpRequestMethod,
            :headers as Dictionary,
            :responseType as HttpResponseContentType,
            :context as Object?,
            :maxBandwidth as Number,
            :fileDownloadProgressCallback as Method(totalBytesTransferred as Number, fileSize as Number?) as Void
};

public function initialize(host, usr, pwd) {

    me.host = ((host == null) || host.equals("")) ? null : proto(host);
    me.usr = usr;
    me.pwd = pwd;

    html_opts = {
        :headers => {
            "Content-Type" => Communications.REQUEST_CONTENT_TYPE_JSON
        },
        :responseType => Communications.HTTP_RESPONSE_CONTENT_TYPE_JSON
    };
}

public function get_host()
{

    return host;
}

public function set_host(host)
{

    me.host = host;
}

private function proto(h)
{

    if (h == null) {
        return h; }

    if (h.find("?") == 0) {
        return null; }

    if (h.find("://") != null) {
        return h; }

    return "https://" + h;
}

public function get(uri, body, cb)
{

    return do_req(Communications.HTTP_REQUEST_METHOD_GET, uri, body, cb);
}

public function post(uri, body, cb)
{

    return do_req(Communications.HTTP_REQUEST_METHOD_POST, uri, body, cb);
}

public function do_req(type, uri, body, cb)
{

    if (host == null || host.length() == 0) {
        //C.set_alert(null, "No host");
        return;
    }

    me.uri = uri;
    me.body = body;
    me.cb = cb;

    html_opts[:method] = type;
    //L.dbg_log(L.DBG_HTTP, "do_req(" + type + "):" + host + uri + " => " + body);
    Communications.makeWebRequest(
        host + uri,
        body,
        html_opts,
        method(:got_api)
    );
    return;
}

private function b2hex(b as ByteArray)
{

    var str = "";
    for (var i = 0; i < b.size(); ++i) {
        str += b[i].format("%02x"); }
    return str;
}

private function MD5(plain)
{

    var bytes = StringUtil.convertEncodedString(
                    plain,
                    {:fromRepresentation => StringUtil.REPRESENTATION_STRING_PLAIN_TEXT,
                     :toRepresentation => StringUtil.REPRESENTATION_BYTE_ARRAY });
    var hash = new Hash({ :algorithm => Cryptography.HASH_MD5 });
    hash.update(bytes);
    return b2hex(hash.digest());
}

public function auth_body(json as Dictionary<String, Object?>)
{

    rlm = json["rlm"];
    var snonce = json["nnc"];
    var cnonce = b2hex(Cryptography.randomBytes(8));

    var ha1 = MD5(usr + ":" + rlm + ":" + pwd);
    var ha2 = MD5(ha1 + ":" + snonce + ":" + cnonce);
    return {
        "rlm"  => rlm,
        "nnc"  => snonce,
        "cnnc" => cnonce,
        "hash" => ha2,
        "usr"  => usr
    };
}

//! Receive the data from the initial API request
public function got_api(responseCode as Number, data as Dictionary<String, Object?> or PersistedContent.Iterator or String or Null) as Void
{

    //L.pr_json(L.DBG_HTTP, "got_req:" + responseCode, data);

    if (responseCode != 401) {
        cb.invoke(responseCode, data);
        return;
    }

    var auth = auth_body(data);

    Communications.makeWebRequest(
        host + "/api/auth/login",
        auth,
        {
            :method => Communications.HTTP_REQUEST_METHOD_POST,
            :headers => {
                "Content-Type" => Communications.REQUEST_CONTENT_TYPE_JSON
            },
            :responseType => Communications.HTTP_RESPONSE_CONTENT_TYPE_JSON
        },
        method(:got_login)
    );
}

//! Receive the data from the initial API request
public function got_login(responseCode as Number, data as Dictionary or String or Null) as Void
{

    //L.pr_json(L.DBG_HTTP, "Login:" + responseCode, data);

    if (responseCode != 200) {
        C.set_alert(null, "Bad auth");
        return;
    }

    var dict = data as Dictionary<String, String>;
    if (dict["error"] != null) {
        C.set_alert(null, "Bad usr/pwd");
        return;
    }

    html_opts[:headers] = {
        "Content-Type" => Communications.REQUEST_CONTENT_TYPE_JSON,
        "Authorization" => "Bearer " + dict["jwt"]
    };

    Communications.makeWebRequest(
        host + "/api" + uri,
        body,
        html_opts,
        method(:got_reply)
    );
}

//! Receive the data from the initial API request
public function got_reply(responseCode as Number, data as Dictionary<String, Object?> or PersistedContent.Iterator or String or Null) as Void
{

    //L.pr_json(L.DBG_HTTP, "got_reply:" + responseCode, data);
    cb.invoke(responseCode, data);
}

}
